<?php
/**
 * Plugin Name: Vraimony Verify Widget (by GetEvidex)
 * Description: WooCommerce-friendly embed tools for Vraimony Verify (iframe-only). No tracking. No remote scripts.
 * Version: 4.5.6.0
 * Author: GetEvidex
 * Requires at least: 6.0
 * Tested up to: 6.5
 * Requires PHP: 7.4
 * License: MIT
 * License URI: https://opensource.org/license/mit/
 * Text Domain: vraimony-verify
 */

if (!defined('ABSPATH')) { exit; }

// ------------------------
// Sanitizers (conservative)
// ------------------------
function vraimony_sanitize_token($s) {
  $s = (string)$s;
  $s = preg_replace('/[^a-z0-9_\-\.]/i', '', $s);
  return trim($s);
}

function vraimony_sanitize_view($s) {
  $v = strtolower(vraimony_sanitize_token($s));
  return ($v === 'expanded') ? 'expanded' : 'minimal';
}

function vraimony_sanitize_rid($s) {
  $rid = vraimony_sanitize_token($s);
  if (!$rid) return '';
  if (!preg_match('/^[0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12}$/i', $rid)) return '';
  return $rid;
}

// ------------------------
// Policy Snapshot (optional)
// Stores only a URL reference. No fetching.
// ------------------------
const VRAIMONY_OPT_POLICY_URL = 'vraimony_policy_url';
const VRAIMONY_OPT_POLICY_AUTOOPEN = 'vraimony_policy_autofocus';

// Legacy option keys (read-only fallback)
const ECLAVON_OPT_POLICY_URL = 'eclavon_policy_url';
const ECLAVON_OPT_POLICY_AUTOOPEN = 'eclavon_policy_autofocus';

function vraimony_get_policy_url() {
  $url = (string) get_option(VRAIMONY_OPT_POLICY_URL, '');
  if (!$url) $url = (string) get_option(ECLAVON_OPT_POLICY_URL, '');

  $url = (string) vraimony_apply_filters_compat('vraimony_policy_snapshot_url', 'eclavon_policy_snapshot_url', $url);
  $url = trim($url);
  if (!$url) return '';

  $clean = esc_url_raw($url, ['http', 'https']);
  if (!$clean) return '';
  $parts = wp_parse_url($clean);
  $scheme = isset($parts['scheme']) ? strtolower($parts['scheme']) : '';
  if ($scheme !== 'http' && $scheme !== 'https') return '';
  return $clean;
}

function vraimony_get_policy_autofocus() {
  $v = (string) get_option(VRAIMONY_OPT_POLICY_AUTOOPEN, '0');
  if ($v === '' || $v === null) $v = (string) get_option(ECLAVON_OPT_POLICY_AUTOOPEN, '0');
  return $v === '1';
}

function vraimony_add_settings_page() {
  add_options_page(
    'Vraimony Verify',
    'Vraimony Verify',
    'manage_options',
    'vraimony-verify-settings',
    'vraimony_render_settings_page'
  );
}
add_action('admin_menu', 'vraimony_add_settings_page');

// ------------------------
// Start Here + Diagnostics (WP.org-ready, no telemetry)
// ------------------------
function vraimony_register_admin_pages() {
  // Capability gate
  if (!current_user_can('manage_options')) return;

  // Top-level menu: keeps UX discoverable without telemetry.
  add_menu_page(
    'Vraimony Verify',
    'Vraimony',
    'manage_options',
    'vraimony-start',
    'vraimony_render_start_here_page',
    'dashicons-shield',
    56
  );

  add_submenu_page('vraimony-start', 'Start Here', 'Start Here', 'manage_options', 'vraimony-start', 'vraimony_render_start_here_page');
  add_submenu_page('vraimony-start', 'Settings', 'Settings', 'manage_options', 'vraimony-verify-settings', 'vraimony_render_settings_page');
  add_submenu_page('vraimony-start', 'Diagnostics', 'Diagnostics', 'manage_options', 'vraimony-diagnostics', 'vraimony_render_diagnostics_page');
}
add_action('admin_menu', 'vraimony_register_admin_pages', 9);

function vraimony_render_start_here_page() {
  if (!current_user_can('manage_options')) {
    wp_die(__('You do not have permission to access this page.', 'vraimony-verify'));
  }

  // Simple 3-step wizard (no telemetry). Stores choices locally in WP options.
  $opt_psp = 'vraimony_start_psp';
  $opt_pt  = 'vraimony_start_product_type';
  $opt_pa  = 'vraimony_start_private_appendix';

  if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    check_admin_referer('vraimony_start_here_save', 'vraimony_start_nonce');

    $psp = isset($_POST['psp']) ? sanitize_text_field((string)$_POST['psp']) : 'woocommerce_payments';
    $pt  = isset($_POST['product_type']) ? sanitize_text_field((string)$_POST['product_type']) : 'generic';
    $pa  = isset($_POST['private_appendix']) ? '1' : '0';

    // Allowlist to stay deterministic and safe.
    $psp_allow = ['woocommerce_payments','paypal','stripe_generic'];
    $pt_allow  = ['generic','electronics','fashion','high_value'];
    if (!in_array($psp, $psp_allow, true)) $psp = 'woocommerce_payments';
    if (!in_array($pt, $pt_allow, true)) $pt = 'generic';

    update_option($opt_psp, $psp, false);
    update_option($opt_pt, $pt, false);
    update_option($opt_pa, $pa, false);

    echo '<div class="notice notice-success"><p>Start Here settings saved.</p></div>';
  }

  $psp = (string) get_option($opt_psp, 'woocommerce_payments');
  $pt  = (string) get_option($opt_pt, 'generic');
  $pa  = ((string) get_option($opt_pa, '0')) === '1';

  $base = 'https://getevidex.net';
  $offline = esc_url($base . '/offline.html');
  $court = esc_url($base . '/court-mode.html');
  $docs = esc_url($base . '/docs.html');
  $standard = esc_url($base . '/standard.html#protocol');
  $trust = esc_url($base . '/trust-center.html');
  $transparency = esc_url($base . '/transparency.html');
  $downloads = esc_url($base . '/downloads.html#wordpress');
  $startMirror = esc_url($base . '/start-woocommerce.html');

  // Demo mode: creates a real receipt from a deterministic demo hash (no PII) then downloads PDF/PNG/Receipt JSON.
  $demoUrl = $base . '/woo-autopilot.html?demo=1&reason=not_received'
    . '&psp=' . rawurlencode($psp)
    . '&product_type=' . rawurlencode($pt);
  $demoUrl = esc_url($demoUrl);

  echo '<div class="wrap"><h1>Vraimony — Start Here (WooCommerce)</h1>';
  echo '<p><b>No tracking:</b> no analytics cookies, fingerprinting, or per-user identifiers.</p>';
  echo '<p><b>Reality Audit:</b> integrity verification only (tamper-evident). No legal advice. No guarantee of admissibility/court acceptance/outcome. Does not prove truth, identity, or authorship. Verify is the source of truth.</p>';

  echo '<div style="display:flex;gap:10px;flex-wrap:wrap;margin:12px 0 18px;">';
  echo '<a class="button button-primary" target="_blank" rel="noreferrer" href="' . $downloads . '">Downloads (store-first)</a>';
  echo '<a class="button" target="_blank" rel="noreferrer" href="' . $startMirror . '">Start Here (site mirror)</a>';
  echo '<a class="button" target="_blank" rel="noreferrer" href="' . $offline . '">Offline Verify</a>';
  echo '<a class="button" target="_blank" rel="noreferrer" href="' . $court . '">Court Mode</a>';
  echo '<a class="button" target="_blank" rel="noreferrer" href="' . $docs . '">Docs</a>';
  echo '</div>';

  echo '<div style="max-width:900px;border:1px solid rgba(127,127,127,.25);background:#fff;padding:12px;border-radius:12px;margin:0 0 14px;">';
  echo '<p style="margin:0 0 8px;"><b>Vraimony Standard Seal</b> — integrity-only. Learn the Standard, Trust Center, and Daily Transparency Snapshot:</p>';
  echo '<ul style="margin:0 0 8px 18px;">';
  echo '<li><a target="_blank" rel="noreferrer" href="' . $standard . '">Standard (Protocol v0.1)</a></li>';
  echo '<li><a target="_blank" rel="noreferrer" href="' . $trust . '">Trust Center</a></li>';
  echo '<li><a target="_blank" rel="noreferrer" href="' . $transparency . '">Daily Transparency Snapshot</a></li>';
  echo '</ul>';
  echo '<p class="description" style="margin:0;">Daily Transparency Snapshot is an integrity-only daily signed checkpoint. It is NOT Certificate Transparency (RFC 6962) and does not provide inclusion/consistency proofs.</p>';
  echo '</div>';

  echo '<h2>60-second wizard</h2>';
  echo '<form method="post">';
  wp_nonce_field('vraimony_start_here_save', 'vraimony_start_nonce');

  echo '<table class="form-table" role="presentation" style="max-width:900px;">';
  echo '<tr><th scope="row">Step 1) PSP context</th><td>';
  echo '<select name="psp">';
  $psps = [
    'woocommerce_payments' => 'WooPayments',
    'paypal' => 'PayPal',
    'stripe_generic' => 'Stripe (generic)'
  ];
  foreach ($psps as $k => $label) {
    $sel = ($psp === $k) ? ' selected' : '';
    echo '<option value="' . esc_attr($k) . '"' . $sel . '>' . esc_html($label) . '</option>';
  }
  echo '</select>';
  echo '<p class="description">Used only for response templates and checklist guidance. No automation claims.</p>';
  echo '</td></tr>';

  echo '<tr><th scope="row">Step 1b) Product type</th><td>';
  echo '<select name="product_type">';
  $pts = [
    'generic' => 'Generic',
    'electronics' => 'Electronics (serial/accessories)',
    'fashion' => 'Fashion (condition/returns)',
    'high_value' => 'High-value (unboxing)'
  ];
  foreach ($pts as $k => $label) {
    $sel = ($pt === $k) ? ' selected' : '';
    echo '<option value="' . esc_attr($k) . '"' . $sel . '>' . esc_html($label) . '</option>';
  }
  echo '</select>';
  echo '<p class="description">Affects checklist suggestions (e.g., serial hashes for electronics).</p>';
  echo '</td></tr>';

  echo '<tr><th scope="row">Step 2) Private Appendix (local store only)</th><td>';
  $checked = $pa ? ' checked' : '';
  echo '<label><input type="checkbox" name="private_appendix" value="1"' . $checked . '> Enable optional local-only Private Appendix storage (default OFF)</label>';
  echo '<p class="description">Private Appendix is stored only in Woo order meta and is never sent to Verify. Public outputs remain PII-safe by default.</p>';
  echo '</td></tr>';

  echo '</table>';

  echo '<p><button type="submit" class="button button-primary">Save wizard choices</button></p>';
  echo '</form>';

  echo '<h2>Step 3) Generate first bundle (demo mode)</h2>';
  echo '<p>Demo mode creates a real receipt from a deterministic demo hash (no PII) and lets you generate the Public Bundle (PDF/PNG/Receipt JSON) instantly.</p>';
  echo '<p><a class="button button-primary" target="_blank" rel="noreferrer" href="' . $demoUrl . '">Generate first bundle (demo)</a></p>';

  echo '<h2>Next 3 actions</h2>';
  echo '<ol>';
  echo '<li>Open Woo Autopilot and compute completeness for a real dispute reason.</li>';
  echo '<li>Generate the Response Pack: PDF + PNG + Receipt JSON + copyable response text (no evidence ZIP).</li>';
  echo '<li>Print Insert Slip for orders to standardize customer messaging.</li>';
  echo '</ol>';

  echo '<p style="margin-top:14px;">Need local checks? <a href="admin.php?page=vraimony-diagnostics">Open Diagnostics</a>.</p>';
  echo '</div>';
}

// Activation redirect (no tracking). Opens Start Here once.
function vraimony_on_activate() {
  add_option('vraimony_do_activation_redirect', '1', '', false);
}
register_activation_hook(__FILE__, 'vraimony_on_activate');

function vraimony_maybe_activation_redirect() {
  if (!is_admin()) return;
  if ((string) get_option('vraimony_do_activation_redirect', '0') !== '1') return;
  delete_option('vraimony_do_activation_redirect');
  if (!current_user_can('manage_options')) return;
  wp_safe_redirect(admin_url('admin.php?page=vraimony-start'));
  exit;
}
add_action('admin_init', 'vraimony_maybe_activation_redirect');

function vraimony_render_diagnostics_page() {
  if (!current_user_can('manage_options')) {
    wp_die(__('You do not have permission to access this page.', 'vraimony-verify'));
  }

  $plugin_version = '4.5.6.0';
  $wpv = get_bloginfo('version');
  $phpv = PHP_VERSION;
  $woo = class_exists('WooCommerce') ? 'Yes' : 'No';

  echo '<div class="wrap"><h1>Vraimony — Diagnostics</h1>';
  echo '<p class="description">Local checks only. No network calls. No telemetry.</p>';
  echo '<table class="widefat striped" style="max-width:900px;">';
  echo '<tr><th>Plugin version</th><td><code>' . esc_html($plugin_version) . '</code></td></tr>';
  echo '<tr><th>WordPress</th><td><code>' . esc_html($wpv) . '</code></td></tr>';
  echo '<tr><th>PHP</th><td><code>' . esc_html($phpv) . '</code></td></tr>';
  echo '<tr><th>WooCommerce active</th><td><code>' . esc_html($woo) . '</code></td></tr>';
  echo '<tr><th>No tracking</th><td>Confirmed: plugin does not set analytics cookies or user identifiers.</td></tr>';
  echo '<tr><th>Remote scripts</th><td>Confirmed: plugin embeds iframe-only widget; no remote scripts injected into WP pages.</td></tr>';
  echo '<tr><th>Evidence ZIP policy</th><td>Confirmed: outputs are PDF/PNG/Receipt JSON only (no evidence ZIP).</td></tr>';
  echo '</table>';

  echo '<p style="margin-top:14px;">Back to <a href="admin.php?page=vraimony-start">Start Here</a>.</p>';
  echo '</div>';
}

function vraimony_render_settings_page() {
  if (!current_user_can('manage_options')) {
    wp_die(__('You do not have permission to access this page.', 'vraimony-verify'));
  }

  if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    check_admin_referer('vraimony_verify_settings_save', 'vraimony_nonce');

    $url = isset($_POST['policy_url']) ? (string) $_POST['policy_url'] : '';
    $url = trim($url);
    $clean = $url ? esc_url_raw($url, ['http', 'https']) : '';
    update_option(VRAIMONY_OPT_POLICY_URL, $clean ? $clean : '', false);

    $auto = isset($_POST['policy_autofocus']) ? '1' : '0';
    update_option(VRAIMONY_OPT_POLICY_AUTOOPEN, $auto, false);

    echo '<div class="notice notice-success"><p>Settings saved.</p></div>';
  }

  $policy = esc_attr(get_option(VRAIMONY_OPT_POLICY_URL, get_option(ECLAVON_OPT_POLICY_URL, '')));
  $auto = vraimony_get_policy_autofocus();

  echo '<div class="wrap"><h1>Vraimony Verify</h1>';
  echo '<p><b>No tracking:</b> this plugin does not use analytics cookies or user identifiers. It embeds an iframe-only verify widget and safe links.</p>';
  echo '<h2>Policy Snapshot (optional)</h2>';
  echo '<p>Provide a public policy URL reference (returns/warranty/shipping terms). The plugin will <b>not</b> fetch policy content. The URL is passed only as an optional reference.</p>';

  echo '<form method="post">';
  wp_nonce_field('vraimony_verify_settings_save', 'vraimony_nonce');
  echo '<table class="form-table" role="presentation">';
  echo '<tr><th scope="row"><label for="policy_url">Policy URL</label></th>';
  echo '<td><input name="policy_url" id="policy_url" type="url" class="regular-text" value="' . $policy . '" placeholder="https://example.com/returns" />';
  echo '<p class="description">Optional public reference only. No fetching. Stored in WordPress options.</p></td></tr>';

  echo '<tr><th scope="row">Auto-open in Autopilot</th><td>';
  echo '<label><input type="checkbox" name="policy_autofocus" ' . ($auto ? 'checked' : '') . ' /> Auto-open Policy Snapshot section when opening Vraimony tools</label>';
  echo '</td></tr>';
  echo '</table>';

  submit_button('Save');
  echo '</form>';

  echo '<hr />';
  echo '<h2>Agency override</h2>';
  echo '<p>You may override the policy URL per store via the <code>vraimony_policy_snapshot_url</code> filter (legacy: <code>eclavon_policy_snapshot_url</code>).</p>';
  echo '</div>';
}

// ------------------------
// Shortcodes
// ------------------------
// [vraimony_verify_widget rid="..." view="minimal" height="520" width="720"]
function vraimony_verify_widget_shortcode($atts = []) {
  $atts = shortcode_atts([
    'rid' => '',
    'view' => 'minimal',
    'height' => '520',
    'width' => '720'
  ], $atts, 'vraimony_verify_widget');

  $rid = vraimony_sanitize_rid($atts['rid']);
  $view = vraimony_sanitize_view($atts['view']);
  $h = intval($atts['height']);
  $w = intval($atts['width']);
  if ($h < 180) $h = 180;
  if ($h > 720) $h = 720;
  if ($w < 260) $w = 260;
  if ($w > 960) $w = 960;

  $q = 'view=' . rawurlencode($view);
  if ($rid) $q .= '&rid=' . rawurlencode($rid);
  $src = esc_url("https://verify.getevidex.net/widget/index.html?{$q}");

  $html = '<iframe '
    . 'src="' . $src . '" '
    . 'sandbox="allow-scripts allow-same-origin" '
    . 'referrerpolicy="no-referrer" '
    . 'loading="lazy" '
    . 'style="border:0;width:' . esc_attr($w) . 'px;max-width:100%;height:' . esc_attr($h) . 'px;" '
    . 'title="' . esc_attr__('Vraimony Verify', 'vraimony-verify') . '"></iframe>';

  $html .= '<p style="font-size:12px;opacity:.78;margin:.55em 0 0;">Integrity verification only (tamper-evident). No legal advice. No guarantee of admissibility/court acceptance. Does not prove truth, identity, or authorship.</p>';
  return $html;
}

// [vraimony_evidence_bundle rid="..."]
function vraimony_evidence_bundle_shortcode($atts = []) {
  $atts = shortcode_atts([
    'rid' => ''
  ], $atts, 'vraimony_evidence_bundle');

  $rid = vraimony_sanitize_rid($atts['rid']);
  $policy_url = vraimony_get_policy_url();

  $ridParam = $rid ? ('?rid=' . rawurlencode($rid)) : '';
  $policyParam = $policy_url ? ((strpos($ridParam, '?') !== false ? '&' : '?') . 'policy_url=' . rawurlencode($policy_url)) : '';

  $tx = esc_url('https://verify.getevidex.net/transaction-pack.html' . $ridParam . $policyParam);
  $court = esc_url('https://verify.getevidex.net/court-mode.html' . $ridParam . $policyParam);
  $verify = $rid ? esc_url('https://verify.getevidex.net/r/' . rawurlencode($rid)) : esc_url('https://verify.getevidex.net/scan');

  $autopilot = vraimony_get_autopilot_base_url();
  $autopilot .= (strpos($autopilot, '?') !== false) ? '&' : '?';
  if ($rid) $autopilot .= 'rid=' . rawurlencode($rid);
  if ($policy_url) $autopilot .= ($rid ? '&' : '') . 'policy_url=' . rawurlencode($policy_url);

  $html = '<div style="border:1px solid #e5e7eb;border-radius:10px;padding:12px;max-width:720px;">'
    . '<div style="font-weight:600;margin-bottom:6px;">Vraimony Evidence Bundle Autopilot</div>'
    . '<div style="font-size:13px;opacity:.85;margin-bottom:10px;">Court-friendly outputs: 1-page Evidence Sheet (PDF) + Proof Card (PNG) + Portable Receipt JSON. Optional Policy Snapshot reference. Integrity-only.</div>'
    . '<div style="display:flex;gap:8px;flex-wrap:wrap;">'
    . '<a href="' . $tx . '" target="_blank" rel="noreferrer" style="padding:8px 10px;border-radius:8px;background:#111827;color:#fff;text-decoration:none;">Generate Evidence Bundle</a>'
    . '<a href="' . $court . '" target="_blank" rel="noreferrer" style="padding:8px 10px;border-radius:8px;border:1px solid #d1d5db;color:#111827;text-decoration:none;">Open Court Mode</a>'
    . '<a href="' . esc_url($autopilot) . '" target="_blank" rel="noreferrer" style="padding:8px 10px;border-radius:8px;border:1px solid #d1d5db;color:#111827;text-decoration:none;">Policy Snapshot (optional)</a>'
    . '<a href="' . $verify . '" target="_blank" rel="noreferrer" style="padding:8px 10px;border-radius:8px;border:1px solid #d1d5db;color:#111827;text-decoration:none;">Open Verify</a>'
    . '</div>'
    . '<div style="font-size:12px;opacity:.75;margin-top:10px;">Integrity verification only (tamper-evident). No legal advice. No guarantee of admissibility/court acceptance. Does not prove truth, identity, authorship, delivery/condition, or intent. Verify page is the source of truth.</div>'
    . '</div>';

  return $html;
}

// ------------------------
// WooCommerce Order Meta Box (links only)
// Stores only receipt id in order meta.
// ------------------------
const VRAIMONY_ORDER_META_RID = '_vraimony_receipt_id';
const ECLAVON_ORDER_META_RID = '_eclavon_receipt_id';

// Optional local-only private appendix (stored only in Woo order meta).
const VRAIMONY_ORDER_META_PRIVATE_APPENDIX = '_vraimony_private_appendix_json';
const ECLAVON_ORDER_META_PRIVATE_APPENDIX = '_eclavon_private_appendix_json';

function vraimony_get_autopilot_base_url() {
  $base = 'https://getevidex.net/woo-autopilot.html';
  $base = (string) vraimony_apply_filters_compat('vraimony_autopilot_base_url', 'eclavon_autopilot_base_url', $base);
  $base = trim($base);
  if (!$base) $base = 'https://getevidex.net/woo-autopilot.html';
  $clean = esc_url_raw($base, ['http','https']);
  return $clean ? $clean : 'https://getevidex.net/woo-autopilot.html';
}

function vraimony_build_order_ref($order_id) {
  $order_id = intval($order_id);
  $ref = 'wc_' . $order_id;
  $ref = (string) vraimony_apply_filters_compat('vraimony_autopilot_order_ref_builder', 'eclavon_autopilot_order_ref_builder', $ref, $order_id);
  $ref = preg_replace('/[^a-z0-9_\-\.]/i', '', (string)$ref);
  $ref = substr($ref, 0, 64);
  return trim($ref);
}

// Helper: apply_filters with optional extra args, while calling legacy filter too.
function vraimony_apply_filters_compat($newTag, $oldTag, $value, ...$args) {
  $v = apply_filters($newTag, $value, ...$args);
  $v = apply_filters($oldTag, $v, ...$args);
  return $v;
}

function vraimony_add_order_meta_box() {
  if (!class_exists('WooCommerce')) return;
  add_meta_box(
    'vraimony_order_autopilot',
    'Vraimony Evidence Bundle Autopilot',
    'vraimony_render_order_meta_box',
    'shop_order',
    'side',
    'default'
  );
}
add_action('add_meta_boxes', 'vraimony_add_order_meta_box');

function vraimony_get_order_rid($post_id) {
  $rid = (string) get_post_meta($post_id, VRAIMONY_ORDER_META_RID, true);
  if (!$rid) $rid = (string) get_post_meta($post_id, ECLAVON_ORDER_META_RID, true);
  return vraimony_sanitize_rid($rid);
}

function vraimony_render_order_meta_box($post) {
  if (!current_user_can('edit_post', $post->ID)) return;

  $rid = vraimony_get_order_rid($post->ID);
  $order_ref = vraimony_build_order_ref($post->ID);

  wp_nonce_field('vraimony_order_meta_save', 'vraimony_order_meta_nonce');

  echo '<p style="margin:0 0 8px;"><b>Receipt ID</b> (UUID)</p>';
  echo '<input type="text" name="vraimony_receipt_id" value="' . esc_attr($rid) . '" style="width:100%;" placeholder="00000000-0000-4000-8000-000000000000" />';
  echo '<p style="font-size:12px;opacity:.75;margin:6px 0 10px;">Stored in order meta only. Links only. No customer data is sent offsite.</p>';

  $priv = vraimony_get_order_private_appendix($post->ID);
  echo '<hr style="margin:12px 0;">';
  echo '<p style="margin:0 0 6px;"><b>Private Appendix</b> (local store only)</p>';
  echo '<p style="font-size:12px;opacity:.78;margin:0 0 8px;">Optional JSON for internal references (carrier IDs, internal notes). <b>Not sent to Verify.</b> Avoid customer PII by default.</p>';
  echo '<textarea name="vraimony_private_appendix_json" rows="6" style="width:100%;font-family:ui-monospace, SFMono-Regular, Menlo, Monaco, Consolas, Liberation Mono, Courier New, monospace;" placeholder="{
  "carrier_tracking": "...",
  "internal_note": "..."
}">' . esc_textarea($priv) . '</textarea>';
  echo '<p style="font-size:12px;opacity:.75;margin:6px 0 0;">This stays inside WooCommerce order meta. Export/sharing is optional and your responsibility.</p>';


  $base = vraimony_get_autopilot_base_url();
  $policy_url = vraimony_get_policy_url();

  $autopilot = $base;
  $autopilot .= (strpos($autopilot, '?') !== false) ? '&' : '?';
  $autopilot .= 'order_ref=' . rawurlencode($order_ref);
  if ($rid) $autopilot .= '&rid=' . rawurlencode($rid);
  if ($policy_url) $autopilot .= '&policy_url=' . rawurlencode($policy_url);

  $court = 'https://getevidex.net/court-mode.html';
  if ($rid) $court .= '?rid=' . rawurlencode($rid);

  $slip = 'https://getevidex.net/print-insert-slip.html';
  if ($rid) $slip .= '?rid=' . rawurlencode($rid);

  echo '<div style="display:flex;flex-direction:column;gap:6px;margin-top:10px;">';
  echo '<a class="button button-primary" target="_blank" rel="noreferrer" href="' . esc_url($autopilot) . '">Open Autopilot</a>';
  echo '<a class="button" target="_blank" rel="noreferrer" href="' . esc_url($court) . '">Open Court Mode</a>';
  echo '<a class="button" target="_blank" rel="noreferrer" href="' . esc_url($slip) . '">Print Insert Slip</a>';
  echo '</div>';

  echo '<p style="font-size:12px;opacity:.75;margin:10px 0 0;">Integrity-only. No legal advice. No guarantee of admissibility/court acceptance.</p>';
}

function vraimony_get_order_private_appendix($post_id) {
  $raw = (string) get_post_meta($post_id, VRAIMONY_ORDER_META_PRIVATE_APPENDIX, true);
  if (!$raw) $raw = (string) get_post_meta($post_id, ECLAVON_ORDER_META_PRIVATE_APPENDIX, true);
  $raw = trim($raw);
  if (!$raw) return '';
  // Do not attempt to validate content beyond size; merchants may store internal refs.
  if (strlen($raw) > 12000) $raw = substr($raw, 0, 12000);
  return $raw;
}


function vraimony_save_order_meta_box($post_id) {
  if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;
  if (!isset($_POST['vraimony_order_meta_nonce'])) return;
  if (!wp_verify_nonce((string)$_POST['vraimony_order_meta_nonce'], 'vraimony_order_meta_save')) return;
  if (!current_user_can('edit_post', $post_id)) return;

  $rid = isset($_POST['vraimony_receipt_id']) ? (string) $_POST['vraimony_receipt_id'] : '';
  $rid = vraimony_sanitize_rid($rid);

  if ($rid) {
    update_post_meta($post_id, VRAIMONY_ORDER_META_RID, $rid);
    // Backward compatibility: also write legacy key.
    update_post_meta($post_id, ECLAVON_ORDER_META_RID, $rid);
  } else {
    delete_post_meta($post_id, VRAIMONY_ORDER_META_RID);
    delete_post_meta($post_id, ECLAVON_ORDER_META_RID);
  }

  $priv = isset($_POST['vraimony_private_appendix_json']) ? (string) $_POST['vraimony_private_appendix_json'] : '';
  $priv = trim($priv);
  if ($priv) {
    // Stored locally only. No network calls. Avoid customer PII by default.
    $priv = sanitize_textarea_field($priv);
    if (strlen($priv) > 12000) $priv = substr($priv, 0, 12000);
    update_post_meta($post_id, VRAIMONY_ORDER_META_PRIVATE_APPENDIX, $priv);
    update_post_meta($post_id, ECLAVON_ORDER_META_PRIVATE_APPENDIX, $priv);
  } else {
    delete_post_meta($post_id, VRAIMONY_ORDER_META_PRIVATE_APPENDIX);
    delete_post_meta($post_id, ECLAVON_ORDER_META_PRIVATE_APPENDIX);
  }

  do_action('vraimony_autopilot_receipt_saved', $post_id, $rid);
  // Legacy action
  do_action('eclavon_autopilot_receipt_saved', $post_id, $rid);
}
add_action('save_post_shop_order', 'vraimony_save_order_meta_box');

// Register shortcodes (new)
add_shortcode('vraimony_verify_widget', 'vraimony_verify_widget_shortcode');
add_shortcode('vraimony_evidence_bundle', 'vraimony_evidence_bundle_shortcode');

// Backward compatibility shortcodes
add_shortcode('eclavon_verify_widget', 'vraimony_verify_widget_shortcode');
add_shortcode('eclavon_evidence_bundle', 'vraimony_evidence_bundle_shortcode');

