/*
 * Vraimony Verify-only popup
 * HARD LOCKS:
 * - No background polling
 * - No telemetry / tracking
 * - No network calls (except opening verify pages via the browser)
 * - No remote code
 */

const VERIFY_ORIGIN = 'https://verify.getevidex.net';

function qs(sel) {
  return document.querySelector(sel);
}

function isVerifyUrl(u) {
  try {
    const url = new URL(u);
    return url.origin === VERIFY_ORIGIN;
  } catch {
    return false;
  }
}

function normalizeInputToVerifyUrl(input) {
  const v = String(input || '').trim();
  if (!v) return { ok: false, reason: 'Empty input.' };

  // If user pasted a full URL, require it to be on verify.getevidex.net
  if (/^https?:\/\//i.test(v)) {
    if (!isVerifyUrl(v)) {
      return { ok: false, reason: 'For safety, only verify.getevidex.net URLs are allowed.' };
    }
    return { ok: true, url: v };
  }

  // Otherwise treat as receipt id and open verify page with ?id=
  // Keep it permissive but bounded: allow [A-Za-z0-9-] up to 80 chars.
  if (!/^[A-Za-z0-9-]{6,80}$/.test(v)) {
    return { ok: false, reason: 'Invalid receipt ID format.' };
  }
  return { ok: true, url: `${VERIFY_ORIGIN}/verify.html?id=${encodeURIComponent(v)}` };
}

async function sha256Hex(file) {
  const buf = await file.arrayBuffer();
  const digest = await crypto.subtle.digest('SHA-256', buf);
  const bytes = new Uint8Array(digest);
  let hex = '';
  for (const b of bytes) hex += b.toString(16).padStart(2, '0');
  return hex;
}

function openVerify(url) {
  // Open a new tab to verify.getevidex.net. No fetch/XHR performed.
  try {
    if (chrome?.tabs?.create) {
      chrome.tabs.create({ url });
    } else {
      window.open(url, '_blank', 'noopener,noreferrer');
    }
  } catch {
    window.open(url, '_blank', 'noopener,noreferrer');
  }
}

function bind() {
  const input = qs('#ridOrUrl');
  const openStatus = qs('#openStatus');
  const hashOut = qs('#hash');
  const hashStatus = qs('#hashStatus');

  qs('#btnOpen').addEventListener('click', () => {
    const r = normalizeInputToVerifyUrl(input.value);
    if (!r.ok) {
      openStatus.textContent = r.reason;
      openStatus.className = 'muted bad';
      return;
    }
    openStatus.textContent = `Opening: ${r.url}`;
    openStatus.className = 'muted ok';
    openVerify(r.url);
  });

  qs('#btnClear').addEventListener('click', () => {
    input.value = '';
    openStatus.textContent = '—';
    openStatus.className = 'muted';
    qs('#file').value = '';
    hashOut.textContent = '—';
    hashStatus.textContent = '—';
  });

  qs('#file').addEventListener('change', async () => {
    const f = qs('#file').files?.[0];
    if (!f) {
      hashOut.textContent = '—';
      hashStatus.textContent = '—';
      return;
    }
    // Size guard: keep popup responsive.
    const MAX_BYTES = 25 * 1024 * 1024; // 25MB
    if (f.size > MAX_BYTES) {
      hashOut.textContent = '—';
      hashStatus.textContent = 'File too large for popup hashing (max 25MB).';
      hashStatus.className = 'muted bad';
      return;
    }
    hashStatus.textContent = 'Hashing locally…';
    hashStatus.className = 'muted';
    try {
      const hex = await sha256Hex(f);
      hashOut.textContent = hex;
      hashStatus.textContent = 'Done.';
      hashStatus.className = 'muted ok';
    } catch (e) {
      hashOut.textContent = '—';
      hashStatus.textContent = 'Hashing failed.';
      hashStatus.className = 'muted bad';
    }
  });
}

document.addEventListener('DOMContentLoaded', bind);
