#!/usr/bin/env python3
"""Vraimony CLI — Offline verifier (by GetEvidex) (v4.5.0)

Usage:
  python3 evidex_verify.py --file path/to/file --receipt path/to/receipt.json

Verifies:
  1) SHA-256(file) matches receipt.payload.content_hash.value
  2) Ed25519 signature is valid (receipt.signature)

No network calls.
"""

import argparse
import base64
import hashlib
import json
import sys

from cryptography.hazmat.primitives.asymmetric.ed25519 import Ed25519PublicKey


def b64url_decode(s: str) -> bytes:
    s = s.strip()
    pad = '=' * ((4 - (len(s) % 4)) % 4)
    s = s + pad
    return base64.urlsafe_b64decode(s.encode('ascii'))


def sha256_hex(path: str) -> str:
    h = hashlib.sha256()
    with open(path, 'rb') as f:
        for chunk in iter(lambda: f.read(1024 * 1024), b''):
            h.update(chunk)
    return h.hexdigest()


def load_json(path: str):
    with open(path, 'r', encoding='utf-8') as f:
        return json.load(f)


def main():
    ap = argparse.ArgumentParser()
    ap.add_argument('--file', required=True)
    ap.add_argument('--receipt', required=True)
    args = ap.parse_args()

    receipt = load_json(args.receipt)

    try:
        expected = receipt['payload']['content_hash']['value']
        receipt_id = receipt['payload']['receipt_id']
        seal_code = receipt['payload'].get('seal_code', '')
    except Exception:
        print('ERROR: invalid receipt format', file=sys.stderr)
        return 2

    computed = sha256_hex(args.file)
    ok_hash = (computed == expected)

    # Signature verify
    sig = receipt.get('signature', {})
    try:
        pub_raw = base64.b64decode(sig['pubkey_b64'])
        payload_bytes = b64url_decode(sig['payload_b64'])
        sig_bytes = b64url_decode(sig['sig_b64'])

        pk = Ed25519PublicKey.from_public_bytes(pub_raw)
        pk.verify(sig_bytes, payload_bytes)
        ok_sig = True
    except Exception:
        ok_sig = False

    print('GetEvidex v4.3 — Offline verification')
    print(f'Receipt ID: {receipt_id}')
    if seal_code:
        print(f'Seal code: {seal_code}')
    print(f'Hash match: {"YES" if ok_hash else "NO"}')
    print(f'Signature:  {"VALID" if ok_sig else "INVALID"}')

    if not ok_hash or not ok_sig:
        return 1
    return 0


if __name__ == '__main__':
    raise SystemExit(main())
